package peers

import (
	"context"
	"sync"
	"time"

	"github.com/ethereum/go-ethereum/p2p/enr"
	"github.com/libp2p/go-libp2p/core/network"
	"github.com/libp2p/go-libp2p/core/peer"
	ma "github.com/multiformats/go-multiaddr"
)

type PeerConnectionState int

const (
	// PeerDisconnected means there is no connection to the peer.
	PeerDisconnected PeerConnectionState = iota
	// PeerDisconnecting means there is an on-going attempt to disconnect from the peer.
	PeerDisconnecting
	// PeerConnected means the peer has an active connection.
	PeerConnected
	// PeerConnecting means there is an on-going attempt to connect to the peer.
	PeerConnecting
)

type PeerData struct {
	Address                ma.Multiaddr
	Direction              network.Direction
	ConnState              PeerConnectionState
	Enr                    *enr.Record
	NextValidTime          time.Time
	BadResponses           int
	NextBadNodeReleaseTime time.Time
}

type Store struct {
	ctx    context.Context
	peers  map[peer.ID]*PeerData
	config *storeConfig
	sync.RWMutex
}

type storeConfig struct {
	MaxInboundPeers  int
	MaxOutboundPeers int
	MaxPeers         int
	MaxBadResponses  int
}

func NewStore(ctx context.Context, cfg *storeConfig) *Store {
	return &Store{
		ctx:    ctx,
		peers:  make(map[peer.ID]*PeerData),
		config: cfg,
	}
}

// PeerData returns data associated with a given peer, if any.
// Important: it is assumed that store mutex is locked when calling this method.
func (s *Store) PeerData(pid peer.ID) (*PeerData, bool) {
	peerData, ok := s.peers[pid]
	return peerData, ok
}

// PeerDataGetOrCreate returns data associated with a given peer.
// If no data has been associated yet, newly created and associated data object is returned.
// Important: it is assumed that store mutex is locked when calling this method.
func (s *Store) PeerDataGetOrCreate(pid peer.ID) *PeerData {
	if peerData, ok := s.peers[pid]; ok {
		return peerData
	}
	s.peers[pid] = &PeerData{}
	return s.peers[pid]
}

// SetPeerData updates data associated with a given peer.
// Important: it is assumed that store mutex is locked when calling this method.
func (s *Store) SetPeerData(pid peer.ID, data *PeerData) {
	s.peers[pid] = data
}

// DeletePeerData removes data associated with a given peer.
// Important: it is assumed that store mutex is locked when calling this method.
func (s *Store) DeletePeerData(pid peer.ID) {
	delete(s.peers, pid)
}

// Peers returns map of peer data objects.
// Important: it is assumed that store mutex is locked when calling this method.
func (s *Store) Peers() map[peer.ID]*PeerData {
	return s.peers
}

// Config return store config
func (s *Store) Config() *storeConfig {
	return s.config
}
