package com.wuban.tron.explore.fetch;

import com.wuban.tron.explore.domain.TronTransEvent;
import com.wuban.tron.explore.entity.Census;
import com.wuban.tron.explore.handler.AddressBalanceHandler;
import com.wuban.tron.explore.handler.BlockDataHandler;
import com.wuban.tron.explore.domain.TronAccount;
import com.wuban.tron.explore.domain.TronResponseData;
import com.wuban.tron.explore.handler.CensusDataHandler;
import com.wuban.tron.explore.handler.ContractEventHandler;
import lombok.extern.slf4j.Slf4j;

/**
 *  <core>区块服务引擎</core>
 *
 * @author sky
 * @date 2020/11/04
 */
@Slf4j
public class Engine {

    /**
     * 线程池
     */
    private Executor executor;

    /**
     * 监控线程,负责异常重启
     */
    private Thread monitor;

    /**
     * 区块数据猎手
     */
    private BlockDataFetcher<TronResponseData> fetcher;

    /**
     * 区块数据处理者
     */
    private BlockDataHandler handler;

    /**
     * 账户余额猎手
     */
    private AddressBalanceFetcher<TronAccount> addressFetcher;

    /**
     * 账户余额处理者
     */
    private AddressBalanceHandler addressHandler;

    /**
     * 数据统计猎手
     */
    private CensusDataFetcher<Census> censusDataFetcher;

    /**
     * 数据统计处理者
     */
    private CensusDataHandler censusDataHandler;

    /**
     * 合约事件猎手
     */
    private ContractEventFetcher<TronTransEvent> contractEventFetcher;

    /**
     * 合约事件处理者
     */
    private ContractEventHandler contractEventHandler;

    public static final int BLOCK_FETCHER_NUM = 2;

    public static final int BALANCE_FETCHER_NUM = 2;

    public Engine(String name) {
        this.executor = new Executor(name);
        this.monitor = new Thread(() -> {
            while (!Thread.currentThread().isInterrupted()) {
                try {
                    Thread.sleep(1000);
                    if (this.handler != null && this.handler.isShutdown()) {
                        log.error("{} engine shutdown now restart ", name);
                        reload();
                    }
                } catch (final InterruptedException e) {
                    log.error("{} engine monitor interrupted ", name);
                    break;
                } catch (final Exception e) {
                    log.error("{} engine monitor error ", name, e);
                }
            }
        }, name.concat("_monitor"));
        this.monitor.start();
    }

    public synchronized void start() {

        /*
            区块数据抓取、处理
         */
        this.handler = new BlockDataHandler();
        this.fetcher = new BlockDataFetcher(this.handler);
        this.executor.execute(this.handler);
        for (int i = 0; i < BLOCK_FETCHER_NUM; i++) {
            this.executor.execute(this.fetcher);
        }

        /*
            账户余额抓取、处理
         */
        this.addressHandler = new AddressBalanceHandler();
        this.addressFetcher = new AddressBalanceFetcher(this.addressHandler);
        this.executor.execute(this.addressHandler);
        for (int i = 0; i < BALANCE_FETCHER_NUM; i++) {
            this.executor.execute(this.addressFetcher);
        }

        /*
            统计数据抓取、处理
         */
        this.censusDataHandler = new CensusDataHandler();
        this.censusDataFetcher = new CensusDataFetcher<>(this.censusDataHandler);
        this.executor.execute(this.censusDataFetcher);
        this.executor.execute(this.censusDataHandler);

         /*
            合约事件数据抓取、处理
         */
        /*this.contractEventHandler = new ContractEventHandler();
        this.contractEventFetcher = new ContractEventFetcher<>(this.contractEventHandler);
        this.executor.execute(this.contractEventFetcher);
        this.executor.execute(this.contractEventHandler);*/
    }


    public synchronized void reload() {
        this.executor.shutdown();
        this.executor.rebuild();
        start();
    }

}
