package com.wuban.tron.explore.service.impl;

import com.wuban.tron.explore.constant.Constant;
import com.wuban.tron.explore.dao.LastBlockRepository;
import com.wuban.tron.explore.domain.TronResponseArrayData;
import com.wuban.tron.explore.domain.TronResponseData;
import com.wuban.tron.explore.entity.LastBlock;
import com.wuban.tron.explore.entity.example.LastBlockExample;
import com.wuban.tron.explore.service.LastBlockService;
import com.wuban.tron.explore.service.TronService;
import com.wuban.tron.explore.util.BigDecimalUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.connection.StringRedisConnection;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;

/**
 *  <core>区块高度服务接口实现类</core>
 *
 * @author sky
 * @date 2020/11/02
 */
@Service
@Slf4j
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class LastBlockServiceImpl implements LastBlockService {

    private final LastBlockRepository lastBlockRepository;

    private final TronService tronService;

    private final StringRedisTemplate stringRedisTemplate;

    @Override
    public LastBlock getByIdForUpdate(Long id) {
        return lastBlockRepository.selectByIdForUpdate(id);
    }

    /**
     * 获取区块高度信息:并且更新此高度+1
     *
     * @param id 唯一标识
     * @return Long
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long getForUpdate(Long id) {
        LastBlock lastBlock = lastBlockRepository.selectByIdForUpdate(id);
        long curBlockNum = lastBlock.getCurBlockNum() + 1;
        LastBlock obj = new LastBlock();
        obj.setId(lastBlock.getId());
        obj.setCurBlockNum(curBlockNum);
        this.lastBlockRepository.updateCurBlockNumById(obj);
        return curBlockNum;
    }

    /**
     * 获取区块高度信息
     *
     * @return LastBlock
     */
    @Override
    public LastBlock getOneByExample() {
        LastBlockExample example = new LastBlockExample();
        return this.lastBlockRepository.selectOneByExample(example);
    }

    @Override
    public void refresh() {
        Integer latestNum = 1;
        TronResponseArrayData ret = this.tronService.getBlockByLatestNum(latestNum);
        if (ret != null && !CollectionUtils.isEmpty(ret.getBlock())) {
            TronResponseData data = ret.getBlock().get(0);
            LastBlock record = new LastBlock();
            record.setId(1L);
            record.setLastBlockNum(data.getBlock_header().getRaw_data().getNumber());
            this.lastBlockRepository.updateById(record);
            log.info("定时任务-已更新区块高度num:{}",data.getBlock_header().getRaw_data().getNumber());
        }
    }

    /**
     * 根据DB cur_block_num 同步固定批次的区块号
     */
    @Override
    public void sync() {
        LastBlock lastBlock = this.lastBlockRepository.selectOneByExample(null);
        if (lastBlock == null) {
            return;
        }

        /*
            如果redis未缓存blockNum,从DB中获取
         */
        BigDecimal bg;
        BigDecimal diff;
        long startNum;
        long endNum;
        String redisBlockNum = this.stringRedisTemplate.opsForValue().get(Constant.BLOCK_NUM_KEY);
        if (StringUtils.isEmpty(redisBlockNum)) {
            startNum = lastBlock.getCurBlockNum();
            bg = BigDecimalUtil.getAdd(new BigDecimal(lastBlock.getCurBlockNum()), new BigDecimal(Constant.FIFTY_THOUSAND));
        } else {
            startNum = Long.parseLong(redisBlockNum);
            if (startNum - Constant.THRESHOLD > 0) {
                return;
            }
            bg = BigDecimalUtil.getAdd(new BigDecimal(redisBlockNum), new BigDecimal(Constant.FIFTY_THOUSAND));
        }
        diff = BigDecimalUtil.getSubtract(bg, new BigDecimal(lastBlock.getLastBlockNum()));

        // 判断已缓存 + BLOCK_NUM_KEY 是否超过区块的最大高度值
        if (diff.longValue() < 0) {
            this.stringRedisTemplate.opsForValue().set(Constant.BLOCK_NUM_KEY, bg.toPlainString());
            endNum = bg.longValue();
        } else {
            // 替换掉"-"号，相当于去绝对值
            this.stringRedisTemplate.opsForValue().set(Constant.BLOCK_NUM_KEY, lastBlock.getLastBlockNum().toString());
            endNum = lastBlock.getLastBlockNum();
        }

        // 将区块高度同步到redis list中
        this.stringRedisTemplate.executePipelined(new RedisCallback<Object>() {
            @Override
            public Object doInRedis(RedisConnection redisConnection) throws DataAccessException {
                StringRedisConnection conn = (StringRedisConnection)redisConnection;
                for (long i = startNum; i <= endNum; i++) {
                    conn.lPush(Constant.BLOCK_NUM_LIST_KEY, Long.toString(i));
                }
                log.info("sync redis block num start={},end={}", startNum, endNum);
                return null;
            }
        });
    }

}
