package com.wuban.tron.explore.fetch;

import com.wuban.tron.explore.constant.Constant;
import com.wuban.tron.explore.domain.TronAccount;
import com.wuban.tron.explore.entity.Address;
import com.wuban.tron.explore.service.AddressService;
import com.wuban.tron.explore.service.TronService;
import com.wuban.tron.explore.util.SpringContextUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.CollectionUtils;

import java.util.Set;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

/**
 *  <core>账户余额处理者服务接口方法实现类</core>
 *
 * @author sky
 * @date 2020/11/04
 */
@Slf4j
public class AccountBalanceTask extends Job {

    private AddressService addressService = SpringContextUtil.getBean(AddressService.class);

    private static final int SIZE = 50;

    private String name = "AccountBalanceThreadPool";

    private static ThreadPoolExecutor threadPool;

    private TronService tronService = SpringContextUtil.getBean(TronService.class);

    private Set<String> dataSet;

    public AccountBalanceTask() {
    }

    public AccountBalanceTask(Set<String> dataSet) {
        this.dataSet = dataSet;
        initPool();
    }

    public void fetch(String address) {
        TronAccount tronAccount = this.tronService.getAccount(address);
        if (tronAccount != null) {
            if (tronAccount.getBalance() != null) {
                Address obj = new Address();
                obj.setAddress(address);
                obj.setBalance(tronAccount.getBalance());
                this.addressService.updateById(obj);
                log.info("更新账户余额 account:{}", obj.toString());
            }
        }
    }

    @Override
    public boolean execute() {
        if (!CollectionUtils.isEmpty(dataSet)) {
            dataSet.forEach(o ->  threadPool.execute(() -> fetch(o)));
        }

        return true;
    }

    public synchronized void initPool() {
        if (threadPool == null) {
            threadPool =  new ThreadPoolExecutor(SIZE, SIZE * 2, 10L, TimeUnit.SECONDS, new LinkedBlockingQueue<>(),
                    new ExecutorThreadFactory());
            log.info("初始化线程池 name:{} coreSize:{} maxSize:{}", name, SIZE, SIZE * 2);
        }
    }

    /**
     * The match engine thread factory
     */
    private class ExecutorThreadFactory implements ThreadFactory {
        private final ThreadGroup group;
        private final AtomicInteger threadNumber = new AtomicInteger(1);
        private final String namePrefix;

        ExecutorThreadFactory() {
            final SecurityManager s = System.getSecurityManager();
            this.group = (s != null) ? s.getThreadGroup() :
                    Thread.currentThread().getThreadGroup();
            this.namePrefix = "excutor-" + AccountBalanceTask.this.name + "-thread-";
        }

        @Override
        public Thread newThread(final Runnable r) {
            final Thread t = new Thread(this.group, r, this.namePrefix + this.threadNumber.getAndIncrement(), 0);
            if (t.isDaemon()) { t.setDaemon(false); }
            if (t.getPriority() != Thread.NORM_PRIORITY) { t.setPriority(Thread.NORM_PRIORITY); }
            return t;
        }
    }

}
