package com.wuban.tron.explore.service.impl;

import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import com.wuban.tron.explore.constant.Constant;
import com.wuban.tron.explore.constant.PageConstant;
import com.wuban.tron.explore.dao.*;
import com.wuban.tron.explore.domain.*;
import com.wuban.tron.explore.entity.*;
import com.wuban.tron.explore.entity.example.TransactionExample;
import com.wuban.tron.explore.fetch.Executor;
import com.wuban.tron.explore.service.TransactionService;
import com.wuban.tron.explore.util.BigDecimalUtil;
import com.wuban.tron.explore.util.DateUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.connection.StringRedisConnection;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.*;

/**
 *  <core>区块服务接口方法实现类</core>
 *
 * @author sky
 * @date 2020/11/02
 */
@Slf4j
@Service
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class TransactionServiceImpl implements TransactionService {

    private final BlockHeaderRepository blockHeaderRepository;

    private final TransactionRepository transactionRepository;

    private final TransactionHexRepository transactionHexRepository;

    private final BlockDayCensusRepository blockDayCensusRepository;

    private final LastBlockRepository lastBlockRepository;

    private final AddressRepository addressRepository;

    private final StringRedisTemplate stringRedisTemplate;

    private Executor excutor;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(List<TronResponseData> dataList) {

        if (!CollectionUtils.isEmpty(dataList)) {
             List<BlockHeader> headerList = new ArrayList<>(dataList.size());
             List<Transaction> transactionList = new ArrayList<>(dataList.size());
             List<TransactionHex> hexList = new ArrayList<>(dataList.size());
             dataList.forEach(o -> {
                 TronBlockHeader tronBlockHeader = o.getBlock_header();
                 BlockHeader header = transferBlockHeader(o.getBlockID(), tronBlockHeader);
                 List<Transactions> trList = o.getTransactions();
                 if (CollectionUtils.isEmpty(trList)) {
                     header.setTransactionVolume(0);
                 } else {
                     header.setTransactionVolume(trList.size());

                     List<Transaction> list = transferTransaction(o.getBlockID(), trList, header.getNumber());
                     transactionList.addAll(list);

                     // hex设置
                     TransactionHex hex = new TransactionHex();
                     trList.forEach(obj -> {
                         hex.setBlockId(o.getBlockID());
                         hex.setTxId(obj.getTxID());
                         hex.setHex(obj.getRaw_data_hex());
                         hex.setSignature(obj.getSignature().get(0));
                         hex.setCreateTime(new Date());
                         hexList.add(hex);
                     });
                 }

                 // 区块bytes设置
                 if (!CollectionUtils.isEmpty(transactionList)) {
                     String refBlockBytes = transactionList.get(0).getRefBlockBytes();
                     String blockBytes = new BigInteger(refBlockBytes, 16).toString();
                     header.setBlockBytes(blockBytes);
                 } else {
                     header.setBlockBytes("0");
                 }
                 headerList.add(header);
             });

             /*
                区块头、区块交易、hex持久化
              */
             this.blockHeaderRepository.batchInsert(headerList);
             Set<String> set = new HashSet<>();
             if (transactionList.size() != 0) {
                 transactionList.forEach(o -> {
                     if (!StringUtils.isEmpty(o.getOwnerAddress())) {
                         set.add(o.getOwnerAddress());
                     }
                     if (!StringUtils.isEmpty(o.getToAddress())) {
                         set.add(o.getToAddress());
                     }
                     if (!StringUtils.isEmpty(o.getContractAddress())) {
                         set.add(o.getContractAddress());
                     }
                 });
                 this.transactionRepository.batchInsert(transactionList);
                 this.transactionHexRepository.batchInsert(hexList);
             }

             // 更新区块高度
             LastBlock lastBlock = new LastBlock();
             lastBlock.setId(1L);
             lastBlock.setCurBlockNum(headerList.get(0).getNumber());
             this.lastBlockRepository.updateCurBlockNumById(lastBlock);
             log.info("已同步数据区块高度num:{}",headerList.get(0).getNumber());

             List<Address> records = transferAddress(set);
             if (!CollectionUtils.isEmpty(records)) {
                 this.addressRepository.batchInsertOnDuplicateKey(records);
                /* AccountBalanceTask task = new AccountBalanceTask(set);
                 excutor.execute(task);*/
                 addressPushRedis(set);
             }
        }

    }

    private void addressPushRedis(Set<String> set) {
        stringRedisTemplate.executePipelined(new RedisCallback<Object>() {
            @Override
            public Object doInRedis(RedisConnection redisConnection) throws DataAccessException {
                StringRedisConnection conn = (StringRedisConnection)redisConnection;
                set.forEach(o -> conn.sAdd(Constant.ADDRESS_KEY, o));
                return null;
            }
        });
    }

    /**
     * 封装账户地址信息
     *
     * @param set 账户地址
     * @return
     */
    private List<Address> transferAddress(Set<String> set) {
        List<Address> list = new ArrayList<>();

        set.forEach(o -> {
            Address address = Address.builder().address(o).balance(0L).build();
            list.add(address);
        });

        return list;
    }

    /**
     * 封装区块头信息
     *
     * @param blockId 区块ID
     * @param tronBlockHeader 区块头信息
     * @return
     */
    private BlockHeader transferBlockHeader(String blockId, TronBlockHeader tronBlockHeader) {
        BlockHeader header = new BlockHeader();
        header.setBlockId(blockId);

        BlockHeaderRawData rawData = tronBlockHeader.getRaw_data();
        header.setNumber(rawData.getNumber());
        header.setParentHash(rawData.getParentHash());
        header.setTxTrieRoot(rawData.getTxTrieRoot());
        header.setVersion(rawData.getVersion());
        header.setWitnessAddress(rawData.getWitness_address());
        header.setWitnessSignature(tronBlockHeader.getWitness_signature());
        header.setTimestamp(rawData.getTimestamp());
        header.setCreateTime(new Date());
        return header;
    }

    /**
     * 封装区块交易信息
     *
     * @param blockId  区块ID
     * @param transactionsList  交易数据
     * @param number 块高度
     * @return
     */
    private  List<Transaction> transferTransaction(String blockId, List<Transactions> transactionsList, Long number) {
        List<Transaction> retList = new ArrayList<>(transactionsList.size());
        transactionsList.forEach(o -> {
            TransactionsRawData rawData = o.getRaw_data();
            Contract contract = rawData.getContract().get(0);
            ContractParameter contractParameter = contract.getParameter();
            ContractParameterValue val = contractParameter.getValue();
            Transaction tr = Transaction.builder()
                    .blockId(blockId)
                    .txId(o.getTxID())
                    .contractRet(o.getRet().get(0).getContractRet())
                    .expiration(rawData.getExpiration())
                    .timestamp(rawData.getTimestamp())
                    .refBlockBytes(rawData.getRef_block_bytes())
                    .refBlockHash(rawData.getRef_block_hash())
                    .feeLimit(rawData.getFee_limit())
                    .type(contract.getType())
                    .typeUrl(contractParameter.getType_url())
                    .amount(val.getAmount())
                    .data(val.getData())
                    .contractAddress(val.getContract_address())
                    .ownerAddress(val.getOwner_address())
                    .toAddress(val.getTo_address())
                    .createTime(new Date())
                    .number(number)
                    .build();
            retList.add(tr);
        });

        return retList;
    }

    /*public synchronized void initExcutor() {
        if (excutor == null) {
            excutor = new Executor(Constant.EXCUTOR_NAME_ACCOUNT, 3);
            new AccountBalanceTask();
        }
    }*/

    @Override
    public Long getBlockMinTime() {
        return this.blockHeaderRepository.selectBlockMinTime();
    }

    @Override
    public void censusBlockByLastDay() {
        Date nextDate = DateUtil.addDays(new Date(), -1);
        String date = DateUtil.getFormatDate(nextDate,DateUtil.PATTERN_YMD);
        String startDate = date + " 00:00:00";
        String endDate = date + " 23:59:59";
        this.census(date, startDate, endDate);
    }

    @Override
    public void censusBlockByDate(String startDate, String endDate) {
        if (StringUtils.isEmpty(endDate) || (startDate.equals(endDate))) {
            String _startDate = startDate + " 00:00:00";
            String _endDate = startDate + " 23:59:59";
            this.census(startDate, _startDate, _endDate);
            return;
        }

        String date = startDate;
        while (true) {
            String _startDate = date + " 00:00:00";
            String _endDate = date + " 23:59:59";
            this.census(startDate, _startDate, _endDate);

            if (date == endDate) {
                break;
            }

            Date nextDate = DateUtil.addDays(new Date(), 1);
            date = DateUtil.getFormatDate(nextDate,DateUtil.PATTERN_YMD);
        }
    }

    /**
     * 分页查询
     *
     * @param startIndex 页码
     * @param pageSize   每页记录数
     * @param example    检索条件
     * @return 分页列表
     */
    @Override
    public PageInfo<Transaction> getByPageWithCategory(Integer startIndex, Integer pageSize, TransactionExample example) {
        if (startIndex == null) {
            startIndex = PageConstant.DEFAULT_START_INDEX;
        }

        if (pageSize == null) {
            pageSize = PageConstant.DEFAULT_PAGE_SIZE;
        }

        if (pageSize > PageConstant.MAX_PAGE_SIZE) {
            pageSize = PageConstant.MAX_PAGE_SIZE;
        }

        example.setOrderByClause("`timestamp` DESC");
        PageHelper.startPage(startIndex, pageSize);
        List<Transaction> list = this.transactionRepository.selectByPagerEx(example);
        PageInfo<Transaction> pageInfo = new PageInfo<>(list);

        return pageInfo;
    }

    /**
     * 根据条件检索
     *
     * @param example 检索条件
     * @return
     */
    @Override
    public List<Transaction> getByExample(TransactionExample example) {
        return this.transactionRepository.selectByExample(example);
    }

    @Override
    public PageInfo<Transaction> selectListByAddress(String address, int type, Integer startIndex, Integer pageSize) {
        Date date;
        switch (type) {
            case 1:
                date = DateUtil.addDays(new Date(), -7);
                break;
            case 2:
                date = DateUtil.addDays(new Date(), -14);
                break;
            case 3:
                date = DateUtil.addDays(new Date(), -30);
                break;
            default:
                date = DateUtil.addDays(new Date(), -7);
        }
        String str = DateUtil.getFormatDate(date, DateUtil.PATTERN_YMD_HMS);
        long t = DateUtil.getDateFromDateStr(str);

        if (startIndex == null) {
            startIndex = PageConstant.DEFAULT_START_INDEX;
        }

        if (pageSize == null) {
            pageSize = PageConstant.DEFAULT_PAGE_SIZE;
        }

        if (pageSize > PageConstant.MAX_PAGE_SIZE) {
            pageSize = PageConstant.MAX_PAGE_SIZE;
        }

        PageHelper.startPage(startIndex, pageSize);
        List<Transaction> list = this.transactionRepository.selectListByAddress(address, t);
        PageInfo<Transaction> pageInfo = new PageInfo<>(list);

        return pageInfo;
    }

    private void census(String date, String startDate, String endDate) {
        Long startDateSeconds = DateUtil.getDateFromDateStr(startDate);
        Long endDateSeconds =  DateUtil.getDateFromDateStr(endDate);
        BlockDayCensus data = this.blockDayCensusRepository.censusBlockByTime(startDateSeconds, endDateSeconds);
        if (data != null) {
            Long totalSeconds = 24 * 60 * 60L;
            BigDecimal aver = BigDecimalUtil.getDevide(new BigDecimal(totalSeconds), new BigDecimal(data.getGenBlockTotalNum()));
            data.setGenBlockAverTime(aver.longValue());
            data.setCensusDate(date);
            data.setCreateTime(new Date());

            BigDecimal averBlockBytes = BigDecimalUtil.getDevide(new BigDecimal(data.getTotalBlockBytes()), new BigDecimal(data.getGenBlockTotalNum()));
            data.setAverBlockBytes(averBlockBytes.intValue());
            this.blockDayCensusRepository.insert(data);
        }
    }

}
