package com.wuban.tron.explore.controller.v1;

import com.github.pagehelper.PageInfo;
import com.wuban.tron.explore.constant.Constant;
import com.wuban.tron.explore.constant.HomeSearchTypeEnum;
import com.wuban.tron.explore.entity.Address;
import com.wuban.tron.explore.entity.BlockDayCensus;
import com.wuban.tron.explore.entity.BlockHeader;
import com.wuban.tron.explore.entity.Transaction;
import com.wuban.tron.explore.entity.example.AddressExample;
import com.wuban.tron.explore.entity.example.BlockDayCensusExample;
import com.wuban.tron.explore.entity.example.BlockHeaderExample;
import com.wuban.tron.explore.entity.example.TransactionExample;
import com.wuban.tron.explore.param.response.*;
import com.wuban.tron.explore.service.AddressService;
import com.wuban.tron.explore.service.BlockDayCensusService;
import com.wuban.tron.explore.service.BlockHeaderService;
import com.wuban.tron.explore.service.TransactionService;
import com.wuban.tron.explore.util.ApiResponse;
import com.wuban.tron.explore.util.DateUtil;
import com.wuban.tron.explore.util.ResponseKit;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import java.util.ArrayList;
import java.util.List;

/**
 * <core>API接口</core>
 *
 * @author sky
 * @date 2020/11/02
 */
@Slf4j
@RestController
@RequestMapping("/api/explorer/v1/")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class TransactionControllerV1 {

    private final TransactionService transactionService;

    private final BlockDayCensusService blockDayCensusService;

    private final BlockHeaderService blockHeaderService;

    private final AddressService addressService;

    /**
     * 首页 - 搜索
     *
     * @param condition
     * @return
     */
    @RequestMapping(value="indexGetInfoByCondition/{condition}", method = RequestMethod.GET)
    public ApiResponse search(@PathVariable("condition") String condition) {
        if (StringUtils.isEmpty(condition)) {
            return ResponseKit.success();
        }
        condition = StringUtils.trimAllWhitespace(condition);

        /*
            块查询
         */
        if (isNumeric(condition)) {
            ResDataModel<BlockInfoModel> resDataModel = new ResDataModel<>();
            BlockHeaderExample headerExample = new BlockHeaderExample();
            headerExample.createCriteria().andNumberEqualTo(Long.valueOf(condition));
            BlockHeader blockHeader = this.blockHeaderService.getOneByExample(headerExample);
            BlockInfoModel model = BlockInfoModel.getInstance();
            if (blockHeader != null) {
                model.setNumber(blockHeader.getNumber());
                if (blockHeader.getTimestamp() != null) {
                    Long tt = blockHeader.getTimestamp()/1000;
                    model.setTimestamp(tt.toString());
                }
                model.setParentHash(blockHeader.getParentHash());
                model.setTrans_number(blockHeader.getTransactionVolume());
                model.setSize(blockHeader.getBlockBytes());
            }
            resDataModel.setData(model);
            resDataModel.setT(HomeSearchTypeEnum.BLOCK_INFO.getCode());
            return ResponseKit.success(resDataModel);
        }

         /*
            账户地址
         */
        if (condition.length() == Constant.USER_ADDRESS_LEN) {
            ResDataModel<AddessInfoModel> resDataModel = new ResDataModel<>();
            AddessInfoModel model = AddessInfoModel.getInstance();

            AddressExample addressExample = new AddressExample();
            addressExample.createCriteria().andAddressEqualTo(condition);
            Address address = this.addressService.selectOneByExample(addressExample);
            if (address != null) {
                model.setAddress(address.getAddress());
                model.setBalance(address.getBalance()+"");
            }
            resDataModel.setData(model);
            resDataModel.setT(HomeSearchTypeEnum.ADDRESS_INFO.getCode());
            return ResponseKit.success(resDataModel);
        }

        /*
            交易hash
         */
        if (condition.length()== Constant.TX_ID_LEN) {
            TransInfoModel model = TransInfoModel.getInstance();
            TransactionExample example = new TransactionExample();
            example.createCriteria().andTxIdEqualTo(condition);
            List<Transaction> txList = this.transactionService.getByExample(example);
            if (!CollectionUtils.isEmpty(txList)) {
                Transaction trans = txList.get(0);
                model.setHash(condition);
                model.setBlockHash(trans.getBlockId());
                model.setBlockNumber(trans.getNumber().intValue());
                model.setFrom(trans.getOwnerAddress());
                if (!StringUtils.isEmpty(trans.getContractAddress())) {
                    model.setTo(trans.getContractAddress());
                } else {
                    model.setTo(trans.getToAddress());
                }

                if (trans.getTimestamp() != null) {
                    Long tt = trans.getTimestamp()/1000;
                    model.setTimestamp(tt.toString());
                }
            }
            return ResponseKit.success(model);
        }

        return ResponseKit.success();
    }

    /**
     * 首页 - Latest Blocks
     *
     * @return
     */
    @RequestMapping(value="indexGetBlockInfo", method = RequestMethod.GET)
    public ApiResponse lastBlockList() {
        BlockHeaderExample headerExample = new BlockHeaderExample();
        PageInfo<BlockHeader> pageInfo = this.blockHeaderService.getByPagerEx(null, null, headerExample);
        List<BlockInfoModel> list = transferBlockInfoModel(pageInfo);

        return ResponseKit.success(list);
    }

    /**
     * 首页 - All Blocks
     *
     * @return
     */
    @RequestMapping(value="getBlockInfo/{pageNo}/{pageSize}", method = RequestMethod.GET)
    public ApiResponse allBlockList(@PathVariable("pageNo") Integer pageNo, @PathVariable("pageSize") Integer pageSize) {
        BlockHeaderExample headerExample = new BlockHeaderExample();
        PageInfo<BlockHeader> pageInfo = this.blockHeaderService.getByPagerEx(pageNo, pageSize, headerExample);
        List<BlockInfoModel> list = transferBlockInfoModel(pageInfo);

        ResDataModel<BlockInfoModel> model = new ResDataModel<>();
        model.setData(list);
        model.setTotal(Integer.valueOf(pageInfo.getTotal()+""));
        return ResponseKit.success(model);
    }

    private List<BlockInfoModel> transferBlockInfoModel(PageInfo<BlockHeader> pageInfo) {
        List<BlockInfoModel> list = new ArrayList<>();
        if (!CollectionUtils.isEmpty(pageInfo.getList())) {
            pageInfo.getList().forEach(o -> {
                BlockInfoModel model = BlockInfoModel.getInstance();
                model.setNumber(o.getNumber());
                if (o.getTimestamp() != null) {
                    Long tt = o.getTimestamp()/1000;
                    model.setTimestamp(tt.toString());
                }
                model.setParentHash(o.getParentHash());
                model.setHash(o.getBlockId());
                model.setTrans_number(o.getTransactionVolume());
                model.setSize(o.getBlockBytes());
                list.add(model);
            });
        }

        return list;
    }

    /**
     * 首页 - Latest Transactions
     *
     * @return
     */
    @RequestMapping(value="indexGetTxInfo", method = RequestMethod.GET)
    public ApiResponse lastTransList() {
        PageInfo<Transaction> pageInfo = this.transactionService.getByPageWithCategory(null, null, new TransactionExample());

        List<TransInfoModel> modelList = transferTransInfoModel(pageInfo);

        return ResponseKit.success(modelList);
    }

    /**
     * 首页 - All Transactions
     *
     * @return
     */
    @RequestMapping(value="getTxInfo/{pageNo}/{pageSize}", method = RequestMethod.GET)
    public ApiResponse allTransList(@PathVariable("pageNo") Integer pageNo, @PathVariable("pageSize") Integer pageSize) {
        PageInfo<Transaction> pageInfo = this.transactionService.getByPageWithCategory(pageNo, pageSize, new TransactionExample());

        List<TransInfoModel> modelList = transferTransInfoModel(pageInfo);
        ResDataModel<TransInfoModel> resDataModel = new ResDataModel<>();
        resDataModel.setTotal(Integer.valueOf(pageInfo.getTotal()+""));
        resDataModel.setData(modelList);

        return ResponseKit.success(resDataModel);
    }

    private List<TransInfoModel> transferTransInfoModel(PageInfo<Transaction> pageInfo) {
        List<TransInfoModel> modelList = new ArrayList<>();
        List<Long> numberList = new ArrayList<>();
        if (!CollectionUtils.isEmpty(pageInfo.getList())) {
            pageInfo.getList().forEach(o -> {
                TransInfoModel model = TransInfoModel.getInstance();
                model.setHash(o.getBlockId());
                model.setBlockHash(o.getTxId());
                model.setBlockNumber(o.getNumber().intValue());
                model.setFrom(o.getOwnerAddress());
                BlockInfoModel blockInfoModel = BlockInfoModel.getInstance();
                blockInfoModel.setNumber(o.getNumber().intValue());
                model.setBlockNumber(blockInfoModel);
                if (!StringUtils.isEmpty(o.getContractAddress())) {
                    model.setTo(o.getContractAddress());
                } else {
                    model.setTo(o.getToAddress());
                }

                if (o.getTimestamp() != null) {
                    Long tt = o.getTimestamp()/1000;
                    model.setTimestamp(tt.toString());
                }
                modelList.add(model);
                numberList.add(o.getNumber());
            });
        }

        return modelList;
    }

    /**
     * 区块列表-通过区块高度查询区块详情
     *
     * @return
     */
    @RequestMapping(value="getBlockDetailInfoByBlock/{block}", method = RequestMethod.GET)
    public ApiResponse getBlockDetailInfoByBlock(@PathVariable("block") String block) {
        BlockHeaderExample headerExample = new BlockHeaderExample();
        headerExample.createCriteria().andNumberEqualTo(Long.valueOf(block));
        BlockHeader blockHeader = this.blockHeaderService.getOneByExample(headerExample);
        BlockInfoModel model = BlockInfoModel.getInstance();
        if (blockHeader != null) {
            model.setNumber(blockHeader.getNumber());
            if (blockHeader.getTimestamp() != null) {
                Long tt = blockHeader.getTimestamp()/1000;
                model.setTimestamp(tt.toString());
            }
            model.setParentHash(blockHeader.getParentHash());
            model.setTrans_number(blockHeader.getTransactionVolume());
            model.setSize(blockHeader.getBlockBytes());
        }

        return ResponseKit.success(model);
    }

    /**
     * 区块列表-通过区块高度查询交易列表
     *
     * @return
     */
    @RequestMapping(value="getTxInfoByBlock/{block}/{pageNo}/{pageSize}", method = RequestMethod.GET)
    public ApiResponse getTxInfoByBlock(@PathVariable("block") String block, @PathVariable("pageNo") Integer pageNo, @PathVariable("pageSize") Integer pageSize) {
        TransactionExample example = new TransactionExample();
        example.createCriteria().andNumberEqualTo(Long.valueOf(block));
        PageInfo<Transaction> pageInfo = this.transactionService.getByPageWithCategory(pageNo, pageSize, example);

        List<TransInfoModel> modelList = transferTransInfoModel(pageInfo);
        ResDataModel<TransInfoModel> resDataModel = new ResDataModel<>();
        resDataModel.setTotal(Integer.valueOf(pageInfo.getTotal()+""));
        resDataModel.setData(modelList);

        return ResponseKit.success(resDataModel);
    }

    /**
     * 区块列表-通过地址查询账户信息
     *
     * @return
     */
    @RequestMapping(value="getAddressInfoByAddress/{address}", method = RequestMethod.GET)
    public ApiResponse getAddressInfoByAddress(@PathVariable("address") String address) {
        AddessInfoModel model = AddessInfoModel.getInstance();
        AddressExample addressExample = new AddressExample();
        addressExample.createCriteria().andAddressEqualTo(address);
        Address obj = this.addressService.selectOneByExample(addressExample);
        if (obj != null) {
            model.setAddress(obj.getAddress());
            model.setBalance(obj.getBalance()+"");
        }

        return ResponseKit.success(model);
    }

    /**
     * 交易列表-通过交易哈希查询交易详情
     *
     * @return
     */
    @RequestMapping(value="getTxDetailInfoByHash/{hash}", method = RequestMethod.GET)
    public ApiResponse getTxDetailInfoByHash(@PathVariable("hash") String hash) {
        TransInfoModel model = TransInfoModel.getInstance();
        TransactionExample example = new TransactionExample();
        example.createCriteria().andTxIdEqualTo(hash);
        List<Transaction> txList = this.transactionService.getByExample(example);
        if (!CollectionUtils.isEmpty(txList)) {
            Transaction trans = txList.get(0);
            model.setHash(hash);
            model.setBlockHash(trans.getBlockId());
            model.setBlockNumber(trans.getNumber().intValue());
            model.setFrom(trans.getOwnerAddress());
            if (!StringUtils.isEmpty(trans.getContractAddress())) {
                model.setTo(trans.getContractAddress());
            } else {
                model.setTo(trans.getToAddress());
            }

            if (trans.getTimestamp() != null) {
                Long tt = trans.getTimestamp()/1000;
                model.setTimestamp(tt.toString());
            }
        }

        return ResponseKit.success(model);
    }

    /**
     * 地址详情页-交易列表
     *
     * @return
     */
    @RequestMapping(value="getTxInfoByAddress/{address}", method = RequestMethod.GET)
    public ApiResponse getTxInfoByAddress(@PathVariable("address") String address) {
        ResDataModel<TransInfoModel> resDataModel = new ResDataModel<>();
        PageInfo<Transaction> pageInfo = this.transactionService.selectPageByAddress(address, 1, 25);
        if (!CollectionUtils.isEmpty(pageInfo.getList())) {
            List<TransInfoModel> modelList = transferTransInfoModel(pageInfo);
            resDataModel.setTotal(Integer.valueOf(pageInfo.getTotal()+""));
            resDataModel.setData(modelList);
        }

        return ResponseKit.success(resDataModel);
    }

    /**
     * 账户余额排行榜
     *
     * @return
     */
    @RequestMapping(value="getTopAccountsInfo/{pageNo}/{pageSize}", method = RequestMethod.GET)
    public ApiResponse getTopAccountsInfo(@PathVariable("pageNo") Integer pageNo, @PathVariable("pageSize") Integer pageSize) {
        ResDataModel<TransInfoModel> resDataModel = new ResDataModel<>();
        AddressExample example = new AddressExample();
        example.setOrderByClause("balance desc");
        PageInfo<Address> pageInfo = this.addressService.selectByPager(pageNo, pageSize, example);
        resDataModel.setTotal(Integer.parseInt(pageInfo.getTotal()+""));
        List<AccountInfoModel> modelList = new ArrayList<>();
        if (!CollectionUtils.isEmpty(pageInfo.getList())) {
            pageInfo.getList().forEach(o -> {
                AccountInfoModel infoModel = AccountInfoModel.getInstance();
                infoModel.setAddress(o.getAddress());
                infoModel.setBalance(o.getBalance().toString());
                modelList.add(infoModel);
            });
        }

        resDataModel.setData(modelList);

        return ResponseKit.success(resDataModel);
    }

    /**
     * 首页图表
     *
     * @return
     */
    @RequestMapping(value="indexGetChartInfo", method = RequestMethod.GET)
    public ApiResponse indexGetChartInfo() {
        PageInfo<BlockDayCensus> pageInfo = this.blockDayCensusService.getByPageWithCategory(1, 14, new BlockDayCensusExample());

        List<HomeChartModel> modelList = new ArrayList<>();
        if (!CollectionUtils.isEmpty(pageInfo.getList())) {
            List<BlockDayCensus> list = pageInfo.getList();
            HomeChartModel model;
            int id = 1;
            for (int i = 0; i < list.size(); i++) {
                model = new HomeChartModel();
                model.setId(id);
                long time = DateUtil.getDateFromDateStr(list.get(i).getCensusDate(), DateUtil.PATTERN_YMD);
                model.setTime(time);
                model.setCount(list.get(i).getTotalVolume());
                modelList.add(model);
                id++;
            }
        }

        return ResponseKit.success(modelList);
    }

    public final static boolean isNumeric(String s) {
        if (s != null && !"".equals(s.trim())) {
            return s.matches("^[0-9]*$");
        } else {
            return false;
        }
    }

}
