package com.wuban.tron.explore.fetch;

import com.wuban.tron.explore.handler.AddressBalanceHandler;
import com.wuban.tron.explore.handler.BlockDataHandler;
import com.wuban.tron.explore.domain.TronAccount;
import com.wuban.tron.explore.domain.TronResponseData;
import lombok.extern.slf4j.Slf4j;

/**
 *  <core>区块服务引擎</core>
 *
 * @author sky
 * @date 2020/11/04
 */
@Slf4j
public class Engine {

    /**
     * 线程池
     */
    private Executor executor;

    /**
     * 监控线程,负责异常重启
     */
    private Thread monitor;

    /**
     * 区块数据猎手
     */
    private BlockDataFetcher<TronResponseData> fetcher;

    /**
     * 区块数据处理者
     */
    private BlockDataHandler handler;

    /**
     * 账户余额猎手
     */
    private AddressBalanceFetcher<TronAccount> addressFetcher;

    /**
     * 账户余额处理者
     */
    private AddressBalanceHandler addressHandler;

    public static final int FETCHER_NUM = 5;

    public Engine(String name) {
        this.executor = new Executor(name);
        this.monitor = new Thread(() -> {
            while (!Thread.currentThread().isInterrupted()) {
                try {
                    Thread.sleep(1000);
                    if (this.handler != null && this.handler.isShutdown()) {
                        log.error("{} engine shutdown now restart ", name);
                        reload();
                    }
                } catch (final InterruptedException e) {
                    log.error("{} engine monitor interrupted ", name);
                    break;
                } catch (final Exception e) {
                    log.error("{} engine monitor error ", name, e);
                }
            }
        }, name.concat("_monitor"));
        this.monitor.start();
    }

    public synchronized void start() {

        /*
            区块数据抓取、处理
         */
        this.handler = new BlockDataHandler();
        this.fetcher = new BlockDataFetcher(this.handler);
        this.executor.execute(this.handler);
        //int cupNum = ThreadPoolUtil.getCupNum()*2;
        for (int i = 0; i <= FETCHER_NUM; i++) {
            this.executor.execute(this.fetcher);
        }

        /*
            账户余额抓取、处理
         */
        this.addressHandler = new AddressBalanceHandler();
        this.addressFetcher = new AddressBalanceFetcher(this.addressHandler);
        this.executor.execute(this.addressHandler);
        this.executor.execute(this.addressFetcher);
    }

    public synchronized void reload() {
        this.executor.shutdown();
        this.executor.rebuild();
        start();
    }

}
