package com.wuban.tron.explore.controller;

import com.wuban.tron.explore.entity.Contract;
import com.wuban.tron.explore.entity.SolidityVersion;
import com.wuban.tron.explore.entity.example.ContractExample;
import com.wuban.tron.explore.param.request.ContractCompilerRequest;
import com.wuban.tron.explore.param.request.ContractRequest;
import com.wuban.tron.explore.param.response.ContractModel;
import com.wuban.tron.explore.service.ContractCompilerService;
import com.wuban.tron.explore.service.ContractService;
import com.wuban.tron.explore.service.SolidityVersionService;
import com.wuban.tron.explore.service.TronService;
import com.wuban.tron.explore.util.ApiResponse;
import com.wuban.tron.explore.util.BizExceptionEnum;
import com.wuban.tron.explore.util.ResponseKit;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;

/**
 * <core>合约校验API接口</core>
 *
 * @author sky
 * @date 2020/11/30
 */
@Slf4j
@RestController
@RequestMapping("/api/explorer/v1/")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class ContractController {

    private final ContractService contractService;

    private final TronService tronService;

    private final ContractCompilerService contractCompilerService;

    private final SolidityVersionService solidityVersionService;

    @RequestMapping(value="getVersion", method = RequestMethod.GET)
    public ApiResponse getVersion() {
        List<SolidityVersion> list = this.solidityVersionService.selectByExample(null);
        return ResponseKit.success(list);
    }

    @RequestMapping(value="checkContract/{address}", method = RequestMethod.GET)
    public ApiResponse checkContract(@PathVariable("address") String address) {

        if (StringUtils.isEmpty(address)) {
            return ResponseKit.fail(BizExceptionEnum.REQUEST_NULL.getCode(), BizExceptionEnum.REQUEST_NULL.getMessage());
        }

        /*
            按照合约地址查询DB中是否存在此合约
         */
        ContractExample example = new ContractExample();
        example.createCriteria().andContractAddressEqualTo(address);
        Contract con = this.contractService.selectOneByExample(example);

        // 未编译
        int isCompiler = 0;
        if (con != null) {
            isCompiler = 1;
        }

        return ResponseKit.success(isCompiler);
    }

    @RequestMapping(value="getContractInfoById/{address}", method = RequestMethod.GET)
    public ApiResponse getContractInfoById(@PathVariable("address") String address) {
        if (StringUtils.isEmpty(address)) {
            return ResponseKit.fail(BizExceptionEnum.REQUEST_NULL.getCode(), BizExceptionEnum.REQUEST_NULL.getMessage());
        }

        ContractExample example = new ContractExample();
        example.createCriteria().andContractAddressEqualTo(address);
        Contract con = this.contractService.selectOneByExample(example);

        ContractModel model = new ContractModel();
        if (con != null) {
            BeanUtils.copyProperties(con, model, "id");
            model.setId(con.getContractAddress());
            model.setCode(con.getBytecode());
        }

        return ResponseKit.success(model);
    }

    @RequestMapping(value="compileSolidityOne", method = RequestMethod.POST)
    public ApiResponse compileSolidityOne(@RequestBody @Valid ContractModel reqParam) {
        boolean flag = false;

        Contract con = this.tronService.getContract(reqParam.getId());
        if (con != null) {
            if (con.getBytecode().equals(reqParam.getCode())) {
                flag = true;
            }
        }

        return ResponseKit.success(flag);
    }

    @RequestMapping(value="compileSolidity", method = RequestMethod.POST)
    public ApiResponse compileSolidity(@RequestBody @Valid ContractCompilerRequest reqParam) {
        String code = this.contractCompilerService.compiler(reqParam.getName(), reqParam.getValue(), reqParam.getCompiler());
        return ResponseKit.success(code);
    }
















    @Deprecated
    @RequestMapping(value="compileSolidityAll", method = RequestMethod.POST)
    public ApiResponse compileSolidityAll(@RequestBody @Valid ContractRequest reqParam) {

        String address = reqParam.getAddress();
        if (StringUtils.isEmpty(address)) {
            return ResponseKit.fail(BizExceptionEnum.REQUEST_NULL.getCode(),BizExceptionEnum.REQUEST_NULL.getMessage());
        }

        /*
            按照合约地址查询DB中是否存在此合约，如果不存在，调用链上接口API
         */
        ContractExample example = new ContractExample();
        example.createCriteria().andContractAddressEqualTo(address);
        Contract con = this.contractService.selectOneByExample(example);
        if (con != null) {
            return ResponseKit.success(con);
        }

        // 调用链上接口API
        Contract contract = this.tronService.getContract(address);
        if (contract == null) {
            return ResponseKit.fail(BizExceptionEnum.SERVER_ERROR.getCode(),BizExceptionEnum.SERVER_ERROR.getMessage());
        }

        // 编译合约
        String code = this.contractCompilerService.compiler(reqParam.getName(), reqParam.getValue(), reqParam.getCompiler());

        // 校验btyecode是否一致，不一致返回，一致持久化
        if (!contract.getBytecode().equals(code)) {
            return ResponseKit.fail(BizExceptionEnum.SERVER_ERROR.getCode(),BizExceptionEnum.SERVER_ERROR.getMessage());
        }

        this.contractService.insert(contract);

        return ResponseKit.success();
    }

}
