package com.wuban.tron.explore.fetch;

import com.wuban.tron.explore.util.ThreadPoolUtil;
import lombok.extern.slf4j.Slf4j;

import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * @author sky
 *
 */
@Slf4j
public class PersistThreadPool {

    private static final int POLL_SIZE = ThreadPoolUtil.getCupNum();

    public final List<ThreadPoolExecutor> executors = new ArrayList<>(POLL_SIZE);

    public PersistThreadPool() {

        for (int i = 0; i < POLL_SIZE; i++) {
            executors.add(new ThreadPoolExecutor(POLL_SIZE, POLL_SIZE * 2, 10L, TimeUnit.SECONDS,
                new LinkedBlockingQueue<>(), new PersistThreadFactory(i)));
        }
        log.info("初始化线程池 name:{} coreSize:{} maxSize:{}", "PersistThreadPool", POLL_SIZE, POLL_SIZE * 2);
    }

    public void stop() {
        for (final ThreadPoolExecutor executor : executors) {
            ThreadPoolUtil.ensureShutdown(executor);
        }
    }

    public ThreadPoolExecutor getPool() {
        Random random = new Random();
        int index = random.nextInt(POLL_SIZE);
        return executors.get(index);
    }

    /**
     * The persist thread factory
     */
    static class PersistThreadFactory implements ThreadFactory {
        private static final AtomicInteger POOL_NUMBER = new AtomicInteger(1);
        private final ThreadGroup group;
        private final AtomicInteger threadNumber = new AtomicInteger(1);
        private final String namePrefix;

        PersistThreadFactory(final int index) {
            final SecurityManager s = System.getSecurityManager();
            this.group = (s != null) ? s.getThreadGroup() : Thread.currentThread().getThreadGroup();
            this.namePrefix = "persist-pool" + index + "-" + POOL_NUMBER.getAndIncrement() + "-thread-";
        }

        @Override
        public Thread newThread(final Runnable r) {
            final Thread t = new Thread(this.group, r,this.namePrefix + this.threadNumber.getAndIncrement(),0);
            if (t.isDaemon()) { t.setDaemon(false); }
            if (t.getPriority() != Thread.NORM_PRIORITY) { t.setPriority(Thread.NORM_PRIORITY); }
            return t;
        }
    }


}
